<?php

namespace App\Services;

use Illuminate\Support\Facades\Log;

class ZatcaService
{
    protected $baseUrl;
    protected $db;
    protected $username;
    protected $password;
    protected $sessionId = null;
    protected $userId = null;
    protected $cookies = [];

    public function __construct()
    {
        $this->baseUrl = rtrim(config('odoo.base_url'), '/');
        $this->db = config('odoo.db');
        $this->username = config('odoo.username');
        $this->password = config('odoo.password');
    }

    /**
     * Make a cURL request with specified parameters
     *
     * @param string $url The URL to request
     * @param string $method HTTP method (GET, POST)
     * @param array $data Data to send in the request
     * @param array $headers Additional headers
     * @return array Response data and cookies
     */
    protected function makeRequest($url, $method = 'GET', $data = [], $headers = [])
    {
        $curl = curl_init();

        $defaultHeaders = [
            'Content-Type: application/json',
            'Accept: application/json',
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/96.0.4664.110 Safari/537.36',
        ];

        $finalHeaders = array_merge($defaultHeaders, $headers);

        $options = [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => '',
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => $method,
            CURLOPT_HTTPHEADER => $finalHeaders,
            CURLOPT_HEADER => true,
            CURLOPT_SSL_VERIFYPEER => false,  // Disable SSL verification - use with caution
            CURLOPT_SSL_VERIFYHOST => 0,      // Disable host verification - use with caution
        ];

        if ($method === 'POST' && !empty($data)) {
            $options[CURLOPT_POSTFIELDS] = json_encode($data);
        }

        // Add session cookies if available
        if ($this->sessionId) {
            $options[CURLOPT_COOKIE] = "session_id={$this->sessionId}";
        }

        curl_setopt_array($curl, $options);

        $response = curl_exec($curl);
        $err = curl_error($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        $headerSize = curl_getinfo($curl, CURLINFO_HEADER_SIZE);

        curl_close($curl);

        if ($err) {
            throw new \Exception("cURL Error: " . $err);
        }

        // Split header and body
        $header = substr($response, 0, $headerSize);
        $body = substr($response, $headerSize);

        // Parse cookies from headers
        $cookies = [];
        preg_match_all('/^Set-Cookie:\s*([^;]*)/mi', $header, $matches);
        foreach ($matches[1] as $cookie) {
            $cookieParts = explode('=', $cookie, 2);
            if (count($cookieParts) == 2) {
                $cookies[$cookieParts[0]] = $cookieParts[1];
            }
        }

        // Decode the JSON body
        $decodedBody = json_decode($body, true);

        return [
            'status_code' => $httpCode,
            'body' => $decodedBody,
            'cookies' => $cookies
        ];
    }

    /**
     * Authenticate with Odoo and get session info
     */
    public function authenticate()
    {
        try {
            $url = "{$this->baseUrl}/web/session/authenticate";
            $data = [
                'jsonrpc' => '2.0',
                'params' => [
                    'login' => $this->username,
                    'password' => $this->password,
                    'db' => $this->db
                ]
            ];

            $response = $this->makeRequest($url, 'POST', $data);

            // Check if response has errors
            if (isset($response['body']['error'])) {
                $errorMsg = $response['body']['error']['message'] ?? 'Unknown error';
                $errorData = $response['body']['error']['data']['message'] ?? '';
                Log::error('Odoo authentication failed', ['message' => $errorMsg, 'details' => $errorData]);
                throw new \Exception("Authentication failed: {$errorMsg} - {$errorData}");
            }

            $data = $response['body'];

            if (isset($data['result']['uid'])) {
                // Store the complete session info
                $this->userId = $data['result']['uid'];

                // Set session ID from cookies
                if (isset($response['cookies']['session_id'])) {
                    $this->sessionId = $response['cookies']['session_id'];
                    $this->cookies = $response['cookies'];
                    return true;
                }
            }

            Log::error('Odoo authentication failed', ['response' => $data]);
            throw new \Exception('Authentication failed: No valid session or user ID returned');
        } catch (\Exception $e) {
            Log::error('Odoo authentication error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        }
    }

    /**
     * Create a new invoice in Odoo
     */
    public function createInvoice(array $invoiceData)
    {
        if (!$this->sessionId && !$this->authenticate()) {
            throw new \Exception('Failed to authenticate with Odoo');
        }

        try {
            $url = "{$this->baseUrl}/api/account.move";
            $data = [
                'jsonrpc' => '2.0',
                'params' => [
                    'data' => $invoiceData
                ]
            ];

            $headers = [
                'X-Openerp-Session-Id: ' . $this->sessionId,
            ];

            $response = $this->makeRequest($url, 'POST', $data, $headers);
            $data = $response['body'];

            if (isset($data['result'])) {
                return $data['result'];
            }

            Log::error('Odoo invoice creation failed', ['response' => $data]);
            throw new \Exception('Invoice creation failed: ' . ($data['error']['message'] ?? 'Unknown error'));
        } catch (\Exception $e) {
            Log::error('Odoo invoice creation error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        }
    }
    
       /**
     * Create a new refund Invoice in Odoo
     */
    public function refundInvoice(array $invoiceData)
    {
        if (!$this->sessionId && !$this->authenticate()) {
            throw new \Exception('Failed to authenticate with Odoo');
        }

        try {
            $url = "{$this->baseUrl}/api/v1/invoice/manual-refund";
            $data = [
                'jsonrpc' => '2.0',
                'params' => [
                    'data' => $invoiceData
                ]
            ];

            $headers = [
                'X-Openerp-Session-Id: ' . $this->sessionId,
            ];

            $response = $this->makeRequest($url, 'POST', $data, $headers);
            $data = $response['body'];

            if (isset($data['result'])) {
                   return $data['result']['refund_invoice_id'];
            }

            Log::error('Odoo invoice creation failed', ['response' => $data]);
            throw new \Exception('Invoice creation failed: ' . ($data['error']['message'] ?? 'Unknown error'));
        } catch (\Exception $e) {
            Log::error('Odoo invoice creation error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        }
    }


    /**
     * Confirm an invoice in Odoo
     */
    public function confirmInvoice(int $invoiceId)
    {
        if (!$this->sessionId && !$this->authenticate()) {
            throw new \Exception('Failed to authenticate with Odoo');
        }

        try {
            $url = "{$this->baseUrl}/api/confirm_invoice/{$invoiceId}";
            $data = [
                'params' => []
            ];

            $headers = [
                'X-Openerp-Session-Id: ' . $this->sessionId,
            ];

            $response = $this->makeRequest($url, 'POST', $data, $headers);
            $data = $response['body'];

            if (isset($data['result']['success']) && $data['result']['success']) {
                return $data['result']['QRcode'];
            }

            Log::error('Odoo invoice confirmation failed', ['response' => $data]);
            throw new \Exception('Invoice confirmation failed: ' . ($data['error']['message'] ?? 'Unknown error'));
        } catch (\Exception $e) {
            Log::error('Odoo invoice confirmation error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        }
    }

    /**
     * Get invoice details from Odoo
     */
    public function getInvoice($invoiceId)
    {
        if (!$this->sessionId && !$this->authenticate()) {
            throw new \Exception('Failed to authenticate with Odoo');
        }

        try {
            $url = "{$this->baseUrl}/api/account.move/{$invoiceId}?query={id, name, invoice_line_ids{product_id,name, quantity,price_unit,analytic_distribution},l10n_sa_qr_code_str}";

            $headers = [
                'X-Openerp-Session-Id: ' . $this->sessionId,
            ];

            $response = $this->makeRequest($url, 'GET', [], $headers);
            $data = $response['body'];

            if (isset($data)) {
                return $data;
            }

            Log::error('Odoo invoice retrieval failed', ['response' => $data]);
            throw new \Exception('Invoice retrieval failed: ' . ($data['error']['message'] ?? 'Unknown error'));
        } catch (\Exception $e) {
            Log::error('Odoo invoice retrieval error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            throw $e;
        }
    }

    /**
     * Helper method to create a POS invoice
     */
    public function createPosInvoice($partnerId, $productId, $quantity, $priceUnit, $journalId, $teamId, $analyticDistribution)
    {
        $invoiceData = [
            'partner_id' => $partnerId,
            'move_type' => 'out_invoice',
            'journal_id' => $journalId,
            'team_id' => $teamId,
            'invoice_line_ids' => [
                [
                    0, 0, [
                    'product_id' => $productId,
                    'name' => 'Pos item',
                    'quantity' => $quantity,
                    'price_unit' => $priceUnit,
                    'analytic_distribution' => $analyticDistribution
                ]
                ]
            ]
        ];

        return $this->createInvoice($invoiceData);
    }

    /**
     * Get current session info
     */
    public function getSessionInfo()
    {
        if (!$this->sessionId && !$this->authenticate()) {
            throw new \Exception('No active session');
        }

        return [
            'session_id' => $this->sessionId,
            'user_id' => $this->userId,
            'cookies' => $this->cookies
        ];
    }
}